function sigma = mrci_JT_substitution(gradU,Bz0,Bz,recon_parameters)
% This function is the part of magnetic resonance conductivity imaging (mrci)
% toolbox calulating the equivelent isotropic conductivity using transversal 
% J-substitution algorithm.
% For details of the algorithm please see the below reference,
% Nam H. S. et al. Phys. Med. Biol., 52, 2007.  
% Writen by Saurav Z. K. Sajib, PhD. 
% Last date of modify [yyyy-mm-dd]: 2017-01-20.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% If you use this function please cite the below article:
% Sajib S. Z. K. et al, "Software Toolbox for Low-frequency Conductivity
% and Current Density Imaging using MRI," article submitted to IEEE Trans. Biomed. Eng.
% For any suggestion feel free to contact at iirc.mrci@gmail.com.

% Inputs of the function is,
% gradU[MN3E]: calculated two-dimensional gradient of voltage where,
% ux = squeeze(gradU(:,:,1,:));
% uy = squeeze(gradU(:,:,2,:)); 
% uz = squeeze(gradU(:,:,3,:));
% Bz0[MNE]: calculated Bz data.
% Bz[MNE]:  measured Bz data.
% recon_parameters.sigma0[MN]: conductivity image of the previous iteration. 
% The default value 1 [S/m] at the begining of iteration.
% recon_parameters.VoxelSize[13]: voxel size in millimeter.
% recon_parameters.Mask[MN]: region of interest for (optional).
% Output of the function is,
% sigma[MN]: reconstructed conductivity image.

%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
if isempty(recon_parameters.Mask)
    Mask = double(squeeze(gradU(:,:,1,1))~=0);
else 
    Mask = recon_parameters.Mask;
end
VoxelSize = recon_parameters.VoxelSize;
imSize    = size(Mask,1);
FOV       = VoxelSize(1)*imSize;
if size(recon_parameters.sigma0,1) == 1
    sigma0 = recon_parameters.sigma0.*Mask;
else
    sigma0 = recon_parameters.sigma0;
end    
%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
SizegradU = size(squeeze(gradU));
SizeB  = size(Bz);
SizeB0 = size(Bz0);
if SizegradU(3) < 2
    error('Check the data size of gradient of U ([MN3E]).');
end
if SizeB ~= SizeB0
    error('Check the data size ([MNE]).');
end
if SizegradU(3) == 2
    gradU(:,:,3,:) = zeros(SizegradU(1),SizegradU(2),SizegradU(4));
end
%%%%%%%%%%%%%%%%%%%%% Reconstructing the conductivity %%%%%%%%%%%%%%%%%%%%%
imSize    = size(Mask,1);
sigma  = zeros(imSize,imSize);
Bz_x = zeros(size(Bz));
Bz_y = zeros(size(Bz));
Bz_z = zeros(size(Bz));
kk = 1;
while kk<=SizeB(3)
    [Bz_x(:,:,kk),Bz_y(:,:,kk)] = mk_gradient2D(Bz(:,:,kk)-Bz0(:,:,kk),FOV,imSize);
    kk = kk+1;
end
ux = squeeze(gradU(:,:,1,:)); 
uy = squeeze(gradU(:,:,2,:)); 
uz = squeeze(gradU(:,:,3,:));
clear gradU;
mu0 = 4*pi*1e-7;
for x = 1:imSize
    for y = 1:imSize
        if Mask(x,y) == 1
            gradU  = [squeeze(ux(x,y,:)),squeeze(uy(x,y,:)),squeeze(uz(x,y,:))];
            gradBz = [squeeze(Bz_y(x,y,:)),-1*squeeze(Bz_x(x,y,:)),squeeze(Bz_z(x,y,:))]';
            dot_prod1 = gradU*gradBz;
            dot_prod2 = gradU*gradU';
            dot_prod1 = diag(dot_prod1); dot_prod1 = sum(dot_prod1);
            dot_prod2 = diag(dot_prod2); dot_prod2 = sum(dot_prod2);
            tem = -(dot_prod1/dot_prod2); tem = (1/mu0)*tem;
            tem(isnan(tem)|isinf(tem)) = 0;
            sigma(x,y) = sigma0(x,y)+tem;
            clear gradU J dot_prod1 dot_prod2 tem;
        elseif Mask(x,y) == 0
            sigma(x,y) = 0;
        end
    end
    clear y;
end
clear x;
end

function [data_x,data_y] = mk_gradient2D(data,FOV,imSize)
grad_data = mk_grad_ft(data,imSize,FOV);
data_x = squeeze(grad_data(2,:,:));
data_y = -1*squeeze(grad_data(1,:,:));
clear grad_data;
end