function Jp = mrci_projected_current_density2(J0,Bz0,Bz,recon_parameters)
% This function is the part of magnetic resonance conductivity imaging (mrci)
% toolbox calulating the projected current density (type-2) in MREIT. 
% For details of the algorithm please see the below reference,
% Jeong W. C. et al. IEEE Trans. Biomed. Eng., 61, 2014.  
% Writen by Saurav Z. K. Sajib, PhD. 
% Last date of modify [yyyy-mm-dd]: 2017-01-20.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% If you use this function please cite the below article:
% Sajib S. Z. K. et al, "Software toolbox for low-frequency conductivity
% and current density imaging using MRI," article submitted to IEEE Trans. Biomed. Eng.
% For any suggestion feel free to contact at iirc.mrci@gmail.com.
% Inputs of the function is,
% J0[MN3]: calculated current density from a homogeneous domain where,
% J0x = J0(:,:,1), J0y = J0(:,:,2), and J0z = J0(:,:,3).
% Bz0[MN]: calculated Bz from a homogeneous domain.
% Bz[MN]:  measured Bz data.
% recon_parameters.VoxelSize[13]: voxel size in millimeter.
% recon_parameters.Mask[MN]: region of interest (optional).
% Output of the function,
% Jp[MN3]: computed projected current density where,
% Jpx = Jp(:,:,1), Jpy = Jp(:,:,2), and Jpz = Jp(:,:,3).

%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
if isempty(recon_parameters.Mask)
    Mask = double(J0(:,:,1)~=0);
else 
    Mask = recon_parameters.Mask;
end
VoxelSize = recon_parameters.VoxelSize; 
imSize    = size(Bz,1);
FOV       = VoxelSize(1)*imSize;

J0x = squeeze(J0(:,:,1));
J0y = squeeze(J0(:,:,2));
J0z = squeeze(J0(:,:,3));

[Bz_x,Bz_y] = mk_gradient2D(Bz-Bz0,FOV,imSize);% Gradient calculation 

Bz_x = Mask.*Bz_x;
Bz_y = Mask.*Bz_y;

%%%%%%%%%%%%%%%%%%%%%%%% Projected current density %%%%%%%%%%%%%%%%%%%%%%%%
mu0 = 4*pi*1e-7;
Jp(:,:,1) = Mask.*J0x+(1/mu0)*Bz_y;
Jp(:,:,2) = Mask.*J0y-(1/mu0)*Bz_x;
Jp(:,:,3) = Mask.*J0z;
end

function [data_x,data_y] = mk_gradient2D(data,FOV,imSize)
grad_data = mk_grad_ft(data,imSize,FOV);
data_x = squeeze(grad_data(2,:,:));
data_y = -1*squeeze(grad_data(1,:,:));
clear grad_data;
end