function rJp = mrci_regional_projected_current_density(J0,Bz0,Bz,ROI,...
    recon_parameters)
% This function is the part of magnetic resonance conductivity imaging (mrci)
% toolbox calulating the projected current density (type-1) in MREIT. 
% For details of the algorithm please see the below reference,
% Sajib S.Z.K. et al. Phys. Med. Biol., 57, 2012.  
% Writen by Saurav Z. K. Sajib, PhD. 
% Last date of modify [yyyy-mm-dd]: 2017-01-20.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% If you use this function please cite the below article:
% Sajib S. Z. K. et al, "Software Toolbox for Low-frequency Conductivity
% and Current Density Imaging using MRI," article submitted to IEEE Trans. Biomed. Eng.
% For any suggestion feel free to contact at iirc.mrci@gmail.com.

% Inputs of the function is,
% J0[MN3]: calculated current density from a homogeneous domain where,
% J0x = J0(:,:,1), J0y = J0(:,:,2), and J0z = J0(:,:,3).
% Bz0[MN]: calculated Bz from a homogeneous domain.
% ROI[MN]: region of interest.
% Bz[MN]: measured Bz data.
% recon_parameters.VoxelSize[13]: voxel size in millimeter.
% Output of the function,
% rJp[MN3]: computed regional projected current density within the ROI 
% where, rJpx = Jp(:,:,1), rJpy = Jp(:,:,2), and rJpz = Jp(:,:,3).

%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
VoxelSize = recon_parameters.VoxelSize;
imSize    = size(Bz,1);
FOV       = VoxelSize(1)*imSize;

J0x = squeeze(J0(:,:,1));
J0y = squeeze(J0(:,:,2));
J0z = squeeze(J0(:,:,3));

mu0 = 4*pi*1e-7;
lap_bz = mk_laplacian2D(Bz,FOV,imSize); %Laplcian of Bz
lap_bz = (1/mu0)*lap_bz;

BndVal = ROI.*(Bz-Bz0);
BndVal = (1/mu0)*BndVal;
%Poisson equation 
beta = mrci_poisson_solver2D(lap_bz,VoxelSize,BndVal,ROI);
[beta_x,beta_y] = mk_gradient2D(beta,FOV,imSize); %Gradient calculation
%%%%%%%%%%%%%%%%%%%%%%%% Projected current density %%%%%%%%%%%%%%%%%%%%%%%%
rJp(:,:,1) = ROI.*J0x+beta_y;
rJp(:,:,2) = ROI.*J0y-beta_x;
rJp(:,:,3) = ROI.*J0z;
end

function har2 = mk_laplacian2D(data,FOV,imSize)
grad_data = mk_grad_ft(data,imSize,FOV);
data_x = squeeze(grad_data(2,:,:));
data_y = -1*squeeze(grad_data(1,:,:));
clear grad_data;
grad_data_x = mk_grad_ft(data_x,imSize,FOV);
data_xx = squeeze(grad_data_x(2,:,:));
clear grad_data_x;
grad_data_y = mk_grad_ft(data_y,imSize,FOV);
data_yy = -1*squeeze(grad_data_y(1,:,:));
clear grad_data_y;
har2 = data_xx+data_yy;
end

function [data_x,data_y] = mk_gradient2D(data,FOV,imSize)
grad_data = mk_grad_ft(data,imSize,FOV);
data_x = squeeze(grad_data(2,:,:));
data_y = -1*squeeze(grad_data(1,:,:));
clear grad_data;
end