function sigma = mrci_harmonic_Bz(gradU,Bz,recon_parameters)
% This function is the part of magnetic resonance conductivity imaging (mrci)
% toolbox calulating the equivelent isotropic conductivity using harmonic Bz algorithm. 
% For details of the algorithm please see the below reference,
% Oh S. H. et al. Phys. Med. Biol., 48, 2003.  
% Writen by Saurav Z. K. Sajib, PhD. 
% Last date of modify [yyyy-mm-dd]: 2017-01-20.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% If you use this function please cite the below article:
% Sajib S. Z. K. et al, "Software Toolbox for Low-frequency Conductivity
% and Current Density Imaging using MRI," article submitted to IEEE Trans. Biomed. Eng.
% For any suggestion feel free to contact at iirc.mrci@gmail.com.
% Inputs of the function is,
% gradU[MN2E]: calculated two-dimensional gradient of voltage where,
% ux = squeeze(gradU(:,:,1,:)); uy = squeeze(gradU(:,:,2,:));
% Bz[MNE]: measured Bz data.
% recon_parameters.VoxelSize[13]: voxel size in millimeter.
% recon_parameters.Mask[MN]: region of interest (optional).
% recon_parameters.s0[scaler]: boundary condition for solving Poisson' equation (optional). 
% The default value is 1 [S/m].
% recon_parameters.Lambda: regularization parameter(optional). 
% The default value is 1.
% recon_parameters.PixelSmoothFactor: number of boundary pixels for smoothing (optional). 
% The default value is 0.
% Output of the function is,
% sigma[MN]: reconstructed conductivity image.

%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
if isempty(recon_parameters.Mask)
    Mask = double(squeeze(gradU(:,:,1,1))~=0);
else 
    Mask = recon_parameters.Mask;
end
VoxelSize = recon_parameters.VoxelSize;
imSize    = size(Mask,1);
FOV       = VoxelSize(1)*imSize;
%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
SizegradU = size(squeeze(gradU));
SizeB = size(Bz);
if SizegradU(3) ~= 2
    error('Size of gradient U must be [MN2E]');
end
if SizegradU(4) ~= SizeB(3)
    error('Number of projection is not same.');
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%% Boundary smoothing %%%%%%%%%%%%%%%%%%%%%%%%%%%
if recon_parameters.PixelSmoothFactor~=0
   roi = mk_boundary_smoothing(Mask,recon_parameters.PixelSmoothFactor);
else
    roi = Mask;
end
%%%%%%%%%%%%%%%%%%%%%%% Calculating Laplacian of Bz %%%%%%%%%%%%%%%%%%%%%%%
kk = 1;
LapBz = zeros(size(Bz));
while kk<= SizeB(3)
    LapBz(:,:,kk) = mk_laplacian2D(squeeze(Bz(:,:,kk)),FOV,imSize);
    kk = kk+1;
end
clear kk;
%%%%%%%%%%%%%%%%%%%%% Reconstructing the conductivity %%%%%%%%%%%%%%%%%%%%%
mu0 = 4*pi*1e-7;
Ux = squeeze(gradU(:,:,1,:)); Uy = squeeze(gradU(:,:,2,:));
LapSigma = laplacian_x(roi,Ux,Uy,(1/mu0)*LapBz,FOV,recon_parameters.Lambda);

%Poisson equation 
sigma = mrci_poisson_solver2D(LapSigma,VoxelSize,...
    recon_parameters.s0,roi);
end

function roi = mk_boundary_smoothing(Mask,PixelSmoothFactor)
SizeMask = size(Mask);
MaskBnd = imerode(Mask,strel('disk',1));
MaskBnd = Mask-MaskBnd;
[y,x]   = find(MaskBnd==1);
[x,y]   = rearrange_points(x,y);
curve_smooth = smoothn({x,y},'robust'); clear x y;
x = curve_smooth{1};  y = curve_smooth{2};
x = [x;x(1)];   y = [y;y(1)];
PixelSmoothFactor = 1-PixelSmoothFactor/SizeMask(1);
cx = mean(x(:)); cy = mean(y(:)); 
x = x-cx; y = y-cy; % shifting the corodinates in to center
bnd_vertex(:,1) = cx+PixelSmoothFactor*x;
bnd_vertex(:,2) = cy+PixelSmoothFactor*y;
roi = double(poly2mask(bnd_vertex(:,1),bnd_vertex(:,2),...
    SizeMask(1),SizeMask(2)));
clear cx cy PixelSmoothFactor x y curve_smooth MaskBnd bnd_vertex;
end

function lap_x = laplacian_x(roi,Ux,Uy,b0,FOV,reg_param)
%%%%%%%%%%%%%%%%%%%%%%%%% Garient of conductivity %%%%%%%%%%%%%%%%%%%%%%%%%
imSize = size(Ux,1);
% Initilizing the variables
sigma_x = zeros(imSize,imSize);
sigma_y = zeros(imSize,imSize);
for x = 1:imSize
    for y = 1:imSize
        if roi(x,y) == 1
            A = [squeeze(Uy(x,y,:)),-1*squeeze(Ux(x,y,:))];
            B = squeeze(b0(x,y,:));
            B = A'*B;
            %Regularization parameters
            lambda = 1/abs(det(A'*A));
            if lambda>1
                lambda = 1/lambda;
            end
            lambda = (lambda*reg_param)*eye(size(A'*A));
            A = A'*A+lambda;
            grad_sigma = cgs(A,B);
            sigma_x(x,y) = grad_sigma(1);
            sigma_y(x,y) = grad_sigma(2);
            clear lambda A B grad_sigma;
        elseif roi(x,y) == 0
            sigma_x(x,y) = 0;
            sigma_y(x,y) = 0;
        end
    end
    clear y;
end
clear x;
%%%%%%%%%%%%%%%%%%%%%%%%% Laplacian of conducvity %%%%%%%%%%%%%%%%%%%%%%%%%
Data.x = sigma_x; Data.y = sigma_y;
lap_x = mk_laplacian2D(Data,FOV,imSize,'no');
clear Data;
end

function har2 = mk_laplacian2D(data,FOV,imSize,Option)
if nargin<4 || isempty(Option)
    Option = 'yes';
end
switch Option
    case 'yes'
        grad_data = mk_grad_ft(data,imSize,FOV);
        data_x = squeeze(grad_data(2,:,:));
        data_y = -1*squeeze(grad_data(1,:,:));
        clear grad_data;
    case 'no'
        data_x = data.x;
        data_y = data.y;
end    
grad_data_x = mk_grad_ft(data_x,imSize,FOV);
data_xx = squeeze(grad_data_x(2,:,:));
clear grad_data_x;
grad_data_y = mk_grad_ft(data_y,imSize,FOV);
data_yy = -1*squeeze(grad_data_y(1,:,:));
clear grad_data_y;
har2 = data_xx+data_yy;
end

