function sigma = mrci_noniterative_harmonic_Bz(J,recon_parameters)
% This function is the part of magnetic resonance conductivity imaging (mrci)
% toolbox calulating the equivelent isotropic conductivity using noniterative 
% harmonic Bz algorithm. 
% For details of the algorithm please see the below reference,
% Seo J. K. et al. Inv. Prob., 27, 2011.
% Writen by Saurav Z. K. Sajib, PhD. 
% Last date of modify [yyyy-mm-dd]: 2017-01-20.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% If you use this function please cite the below article:
% Sajib S. Z. K. et al, "Software Toolbox for Low-frequency Conductivity
% and Current Density Imaging using MRI," article submitted to IEEE Trans. Biomed. Eng.
% For any suggestion feel free to contact at iirc.mrci@gmail.com.
% Inputs of the function is,
% J[MN2E]: estimated two-dimensional current density image where,
% Jx = squeeze(J(:,:,1,:)); Jy = squeeze(J(:,:,2,:)).
% recon_parameters.VoxelSize[13]: voxel size in millimeter.
% recon_parameters.Mask[MN]: region of interest (optional).
% recon_parameters.s0[scaler]: boundary condition for solving Poisson's equation (optional). 
% The default value is 1 [S/m].
% recon_parameters.Lambda: regularization parameter(optional). 
% The default value is 1.
% recon_parameters.PxielSmoothFactor: number of boundary pixels for smoothing (optional). 
% The default value is 0.
% Output of the function is,
% sigma[MN]: reconstructed conductivity image.

%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
if isempty(recon_parameters.Mask)
    Mask = double(squeeze(J(:,:,1,1))~=0);
else 
    Mask = recon_parameters.Mask;
end
VoxelSize = recon_parameters.VoxelSize;
imSize    = size(Mask,1);
FOV       = VoxelSize(1)*imSize;
%%%%%%%%%%%%%%%%%%%%%%%%%%%% Boundary smoothing %%%%%%%%%%%%%%%%%%%%%%%%%%%
if recon_parameters.PixelSmoothFactor~=0
   roi = mk_boundary_smoothing(Mask,recon_parameters.PixelSmoothFactor);
else
    roi = Mask;
end
%%%%%%%%%%%%%%%%%% Calculating z-component of curl of J %%%%%%%%%%%%%%%%%%%
kk = 1;
Jx = squeeze(J(:,:,1,:)); Jy = squeeze(J(:,:,2,:));
curl_J = zeros(size(Jx)); 
while kk<= size(J,4)
    curl_J(:,:,kk) = mk_curl_z(squeeze(Jx(:,:,kk)),squeeze(Jy(:,:,kk)),...
        FOV,imSize);
    kk = kk+1;
end
clear kk;
%%%%%%%%%%%%%%%%%%%%% Reconstructing the conductivity %%%%%%%%%%%%%%%%%%%%%
LapSigma = laplacian_x(roi,Jx,Jy,curl_J,FOV,recon_parameters.Lambda);
%Poisson equation 
sigma = mrci_poisson_solver2D(LapSigma,VoxelSize,...
    log(recon_parameters.s0),roi);
sigma = exp(sigma);
sigma = roi.*sigma;
end

function roi = mk_boundary_smoothing(Mask,PixelSmoothFactor)
SizeMask = size(Mask);
MaskBnd = imerode(Mask,strel('disk',1));
MaskBnd = Mask-MaskBnd;
[y,x]   = find(MaskBnd==1);
[x,y]   = rearrange_points(x,y);
curve_smooth = smoothn({x,y},'robust'); clear x y;
x = curve_smooth{1};  y = curve_smooth{2};
x = [x;x(1)];   y = [y;y(1)];
PixelSmoothFactor = 1-PixelSmoothFactor/SizeMask(1);
cx = mean(x(:)); cy = mean(y(:)); 
x = x-cx; y = y-cy; % shifting the corodinates in to center
bnd_vertex(:,1) = cx+PixelSmoothFactor*x;
bnd_vertex(:,2) = cy+PixelSmoothFactor*y;
roi = double(poly2mask(bnd_vertex(:,1),bnd_vertex(:,2),...
    SizeMask(1),SizeMask(2)));
clear cx cy PixelSmoothFactor x y curve_smooth MaskBnd bnd_vertex;
end

function curl_z_data = mk_curl_z(data_x,data_y,FOV,imSize)
[~,data_xy] = mk_gradient2D(data_x,FOV,imSize);
[data_yx,~] = mk_gradient2D(data_y,FOV,imSize);
curl_z_data = data_yx-data_xy;
end

function [data_x,data_y] = mk_gradient2D(data,FOV,imSize)
grad_data = mk_grad_ft(data,imSize,FOV);
data_x =  1*squeeze(grad_data(2,:,:));
data_y = -1*squeeze(grad_data(1,:,:));
clear grad_data;
end

function lap_x = laplacian_x(roi,Ux,Uy,b0,FOV,reg_param)
%%%%%%%%%%%%%%%%%%%%%%%%% Garient of conductivity %%%%%%%%%%%%%%%%%%%%%%%%%
imSize = size(Ux,1);
% Initilizing the variables
sigma_x = zeros(imSize,imSize);
sigma_y = zeros(imSize,imSize);
for x = 1:imSize
    for y = 1:imSize
        if roi(x,y) == 1
            A = [squeeze(Uy(x,y,:)),-1*squeeze(Ux(x,y,:))];
            B = squeeze(b0(x,y,:));
            B = A'*B;A = A'*A;
            lambda = 1/abs(det(A));
            %lambda1(x,y) = lambda;
            lambda = (lambda*reg_param)*eye(size(A));
            A = A+lambda;
            grad_sigma = cgs(A,B);
            sigma_x(x,y) = grad_sigma(1);
            sigma_y(x,y) = grad_sigma(2);
            clear lambda A B grad_sigma;
        end
    end
    clear y;
end
clear x;
%%%%%%%%%%%%%%%%%%%%%%%%% Laplacian of conducvity %%%%%%%%%%%%%%%%%%%%%%%%%
Data.x = sigma_x; Data.y = sigma_y;
lap_x = mk_laplacian2D(Data,FOV,imSize,'no');
clear Data;
end

function har2 = mk_laplacian2D(data,FOV,imSize,Option)
if nargin<4 || isempty(Option)
    Option = 'yes';
end
switch Option
    case 'yes'
        grad_data = mk_grad_ft(data,imSize,FOV);
        data_x = squeeze(grad_data(2,:,:));
        data_y = -1*squeeze(grad_data(1,:,:));
        clear grad_data;
    case 'no'
        data_x = data.x;
        data_y = data.y;
end    
grad_data_x = mk_grad_ft(data_x,imSize,FOV);
data_xx = squeeze(grad_data_x(2,:,:));
clear grad_data_x;
grad_data_y = mk_grad_ft(data_y,imSize,FOV);
data_yy = -1*squeeze(grad_data_y(1,:,:));
clear grad_data_y;
har2 = data_xx+data_yy;
end



