function C = mrci_cti(sigmaH,de,di,alpha,D,beta)

% This function is the part of magnetic resonance conductivity imaging (mrci)
% toolbox calulating the conductivity tensor using newly developed MR-CTI. 
% For details of the algorithm please see the below reference,
% Kwon O. I. et al. "Electrodeless conductivity tensor imaging (CTI) using 
% MRI," Sci. Rep., article under review.
% Writen by Saurav Z. K. Sajib, PhD. 
% Last date of modify [yyyy-mm-dd]: 2017-01-20.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% If you use this function please cite the below article:
% Sajib S. Z. K. et al, "Software Toolbox for Low-frequency Conductivity
% and Current Density Imaging using MRI," article submitted to IEEE Trans. Biomed. Eng.
% For any suggestion feel free to contact at iirc.mrci@gmail.com.
% Inputs of the function is,
% sigmaH[MNNs]: estimated multi-slice high-frequency conductivity images.
% de[MNNs]: estimated multi-slice extracellular water diffusion coefficient images.
% di[MNNs]: estimated multi-slice intracellular water diffusion coefficient images.
% alpha[MNNs]: estimated multi-slice extracellular volume fraction images.
% D[MN6Ns]: measured multi-slice diffusion tensor images.
% beta[MNNs]: estimated multi-slice ion concentration ratio images.
% Output of the function,
% C[MN6Ns]: reconstructed multi-slice conductivity tensor images.

Mask = double(sigma_high~=0);
%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
size_data = size(sigma_high);
if nargin < 5 || isempty(beta)
    beta = 0.7759*Mask;
end
%%%%%%%%%%%%%%%%%% Reconstructing the conductivity tensor %%%%%%%%%%%%%%%%%
alpha1 = 1-alpha;
alpha1 = Mask.*alpha1;
num = alpha(:).*sigmaH(:);
den = alpha(:).*de(:)+beta*alpha1(:).*di(:);
scale_factor = num./den;
scale_factor(isnan(scale_factor)|isinf(scale_factor)) = 0;
scale_factor = reshape(scale_factor,size_data);
scale_factor = Mask.*scale_factor;
C = conductivity_tensor(D,scale_factor,Mask);
end

function CndTensor = conductivity_tensor(D,scale_factor,Mask)

imSize = size(Mask,1);
Nslices = size(Mask,3);
size_scale_factor = size(scale_factor,1);
if size_scale_factor == 1
    scale_factor = scale_factor.*ones(imSize,imSize,Nslices);
end
scale_factor = Mask.*scale_factor;
%%%%%%%%%%%%%%%%%%%%%%%%%%% Conductivity tensor %%%%%%%%%%%%%%%%%%%%%%%%%%%
CndTensor = zeros(size(D));
for z = 1:Nslices
    DTem = squeeze(D(:,:,:,z));
    for x = 1:imSize
        for y = 1:imSize
            dti = [DTem(x,y,1),DTem(x,y,2),DTem(x,y,3);...
                   DTem(x,y,2),DTem(x,y,4),DTem(x,y,5);...
                   DTem(x,y,3),DTem(x,y,5),DTem(x,y,6)];
            dti = Mask(x,y,z).*dti;
            [Vec,Val] = eig(dti);
            Lambda = scale_factor(x,y,z).*Val;
            tem = Vec*Lambda*Vec';
            % Diagonal element
            CndTensor(x,y,1,z) = abs(tem(1,1)); 
            CndTensor(x,y,4,z) = abs(tem(2,2)); 
            CndTensor(x,y,6,z) = abs(tem(3,3));
            % Off-diagonal element
            CndTensor(x,y,2,z) = tem(1,2);      
            CndTensor(x,y,3,z) = tem(1,3);      
            CndTensor(x,y,5,z) = tem(2,3);
            clear dti Vec Val tem Lambda;
        end
    end
    clear Dtem;
end
CndTensor(isnan(CndTensor)|isinf(CndTensor)) = 0;
end

