function [C,ScaleFactor] = mrci_dtmreit_noniterative(J,D,recon_parameters)
% This function is the part of magnetic resonance conductivity imaging (mrci)
% toolbox calulating the equivelent anisotropic conductivity using noniterative 
% DT-MREIT algorithm. 
% For details of the algorithm please see the below reference,
% Kwon O. I. et al. Phys. Med. Biol. 59, 2014.
% Writen by Saurav Z. K. Sajib, PhD. 
% Last date of modify [yyyy-mm-dd]: 2017-01-20.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% If you use this function please cite the below article:
% Sajib S. Z. K. et al, "Software Toolbox for Low-frequency Conductivity
% and Current Density Imaging using MRI," article submitted to IEEE Trans. Biomed. Eng.
% For any suggestion feel free to contact at iirc.mrci@gmail.com.
% Inputs of the function is,
% J[MN3E]: estimated current density image where, 
% Jx = squeeze(J(:,:,1,:)); 
% Jy = squeeze(J(:,:,2,:));
% Jz = squeeze(J(:,:,3,:));
% D[MN6]: water diffusion tensor image.
% recon_parameters.VoxelSize[13]: voxel size in millimeter.
% recon_parameters.Mask[MN]: region of interest (optional).
% recon_parameters.scalefactor0: boundary condition for solving Poisson's equation (optional).  
% The default value is 1 S.sec/mm3.
% recon_parameters.Lambda: regularization parameter (optional). 
% The default value is 1.
% recon_parameters.PxielSmoothFactor: number of boundary pixels for smoothing (optional). 
% The default value is 0.
% Output of the function is,
% C[MN6]: reconstructed conductivity tensor image.
% ScaleFactor[MN]: reconstructed scale factor image.
%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
if isempty(recon_parameters.Mask)
    Mask = double(squeeze(J(:,:,1,1))~=0);
else 
    Mask = recon_parameters.Mask;
end
VoxelSize = recon_parameters.VoxelSize;
imSize    = size(Mask,1);
FOV       = VoxelSize(1)*imSize;
%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
if size(J,4) == 1
    error('Atleast two projection is required.');
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%% Boundary smoothing %%%%%%%%%%%%%%%%%%%%%%%%%%%
if recon_parameters.PixelSmoothFactor~=0
   roi = mk_boundary_smoothing(Mask,recon_parameters.PixelSmoothFactor);
else
    roi = Mask;
end
%%%%%%%%%%%%%%%%%%%%%%% Calculating pJ = D^{-1}*J %%%%%%%%%%%%%%%%%%%%%%%%%
pJx = zeros(imSize,imSize,size(J,4));
pJy = zeros(imSize,imSize,size(J,4));
pJz = zeros(imSize,imSize,size(J,4));
kk = 1;
while kk<=size(J,4)
    [pJx(:,:,kk),pJy(:,:,kk),pJz(:,:,kk)] = mk_pseudo_current_density(D,...
        squeeze(J(:,:,:,kk)),roi,imSize);
    kk = kk+1;
end
%%%%%%%%%%%%%%%%%%%% Calculating z-component of curl %%%%%%%%%%%%%%%%%%%%%%
curl_pJ = zeros(imSize,imSize,size(J,4));
kk = 1;
while kk<= size(J,4)
    curl_pJ(:,:,kk) = mk_curl_z(squeeze(pJx(:,:,kk)),squeeze(pJy(:,:,kk)),...
        FOV,imSize);
    kk = kk+1;
end
clear kk;
%%%%%%%%%%%%%%%%%% Reconstructing the conductivity tensor %%%%%%%%%%%%%%%%%
LapScaleFactor = laplacian_x(roi,pJx,pJy,curl_pJ,FOV,recon_parameters.Lambda);
%Poisson equation
ScaleFactor = mrci_poisson_solver2D(LapScaleFactor,VoxelSize,...
    log(recon_parameters.scalefactor0),roi);
ScaleFactor = exp(ScaleFactor); ScaleFactor = roi.*ScaleFactor;
%Conductivity tensor C = \etaD
C = conductivity_tensor(D,ScaleFactor,Mask);
C = squeeze(C);
end

function [pseudoJx,pseudoJy,pseudoJz] = mk_pseudo_current_density(DTI,J,...
    Mask,imSize)
Jx = squeeze(J(:,:,1));
Jy = squeeze(J(:,:,2));
Jz = squeeze(J(:,:,3));
clear J;
pseudoJx = zeros(imSize,imSize);
pseudoJy = zeros(imSize,imSize);
pseudoJz = zeros(imSize,imSize);
for x = 1:imSize
    for y = 1:imSize
        if Mask(x,y) == 1
            D = [DTI(x,y,1),DTI(x,y,2),DTI(x,y,3);...
                 DTI(x,y,2),DTI(x,y,4),DTI(x,y,5);...
                 DTI(x,y,3),DTI(x,y,5),DTI(x,y,6)];
             J  = [Jx(x,y);Jy(x,y);Jz(x,y)];
             pJ = D\J;
             pseudoJx(x,y) = pJ(1);
             pseudoJy(x,y) = pJ(2);
             pseudoJz(x,y) = pJ(3);
             clear J pJ;
             clear kk D;
        end
    end
    clear y;
end
clear x;
pseudoJx(isnan(pseudoJx)|isinf(pseudoJx)) = 0;
pseudoJy(isnan(pseudoJy)|isinf(pseudoJy)) = 0;
pseudoJz(isnan(pseudoJz)|isinf(pseudoJz)) = 0;
end

function curl_z_data = mk_curl_z(data_x,data_y,FOV,imSize)
[~,data_xy] = mk_gradient2D(data_x,FOV,imSize);
[data_yx,~] = mk_gradient2D(data_y,FOV,imSize);
curl_z_data = data_yx-data_xy;
end

function [data_x,data_y] = mk_gradient2D(data,FOV,imSize)
grad_data = mk_grad_ft(data,imSize,FOV);
data_x = squeeze(grad_data(2,:,:));
data_y = -1*squeeze(grad_data(1,:,:));
clear grad_data;
end

function lap_x = laplacian_x(roi,Ux,Uy,b0,FOV,reg_param)
%%%%%%%%%%%%%%%%%%%%%%%%% Garient of scale factor %%%%%%%%%%%%%%%%%%%%%%%%%
imSize = size(Ux,1);
% Initilizing the variables
eta_x = zeros(imSize,imSize);
eta_y = zeros(imSize,imSize);
for x = 1:imSize
    for y = 1:imSize
        if roi(x,y) == 1
            A = [squeeze(Uy(x,y,:)),-1*squeeze(Ux(x,y,:))];
            B = squeeze(b0(x,y,:));
            %Regularization parameters
            lambda = 1/abs(det(A'*A));
            lambda = (lambda*reg_param)*eye(size(A'*A));
            grad_sigma = cgs(A'*A+lambda,A'*B);
            eta_x(x,y) = grad_sigma(1);
            eta_y(x,y) = grad_sigma(2);
            clear lambda A B grad_sigma;
        end
    end
    clear y;
end
clear x;
%%%%%%%%%%%%%%%%%%%%%%%%% Laplacian of scale factor %%%%%%%%%%%%%%%%%%%%%%%%%
Data.x = eta_x; Data.y = eta_y;
lap_x = mk_laplacian2D(Data,FOV,imSize,'no');
clear Data;
end

function har2 = mk_laplacian2D(data,FOV,imSize,Option)
if nargin<4 || isempty(Option)
    Option = 'yes';
end
switch Option
    case 'yes'
        grad_data = mk_grad_ft(data,imSize,FOV);
        data_x = squeeze(grad_data(2,:,:));
        data_y = -1*squeeze(grad_data(1,:,:));
        clear grad_data;
    case 'no'
        data_x = data.x;
        data_y = data.y;
end    
grad_data_x = mk_grad_ft(data_x,imSize,FOV);
data_xx = squeeze(grad_data_x(2,:,:));
clear grad_data_x;
grad_data_y = mk_grad_ft(data_y,imSize,FOV);
data_yy = -1*squeeze(grad_data_y(1,:,:));
clear grad_data_y;
har2 = data_xx+data_yy;
end
function roi = mk_boundary_smoothing(Mask,PixelSmoothFactor)
SizeMask = size(Mask);
MaskBnd = imerode(Mask,strel('disk',1));
MaskBnd = Mask-MaskBnd;
[y,x]   = find(MaskBnd==1);
[x,y]   = rearrange_points(x,y);
curve_smooth = smoothn({x,y},'robust'); clear x y;
x = curve_smooth{1};  y = curve_smooth{2};
x = [x;x(1)];   y = [y;y(1)];
PixelSmoothFactor = 1-PixelSmoothFactor/SizeMask(1);
cx = mean(x(:)); cy = mean(y(:)); 
x = x-cx; y = y-cy; % shifting the corodinates in to center
bnd_vertex(:,1) = cx+PixelSmoothFactor*x;
bnd_vertex(:,2) = cy+PixelSmoothFactor*y;
roi = double(poly2mask(bnd_vertex(:,1),bnd_vertex(:,2),...
    SizeMask(1),SizeMask(2)));
clear cx cy PixelSmoothFactor x y curve_smooth MaskBnd bnd_vertex;
end

function CndTensor = conductivity_tensor(D,scale_factor,Mask)
imSize = size(Mask,1);
Nslices = size(Mask,3);
size_scale_factor = size(scale_factor,1);
if size_scale_factor == 1
    scale_factor = scale_factor.*ones(imSize,imSize,Nslices);
end
scale_factor = Mask.*scale_factor;
%%%%%%%%%%%%%%%%%%%%%%%%%%% Conductivity tensor %%%%%%%%%%%%%%%%%%%%%%%%%%%
CndTensor = zeros(size(D));
for z = 1:Nslices
    DTem = squeeze(D(:,:,:,z));
    for x = 1:imSize
        for y = 1:imSize
            dti = [DTem(x,y,1),DTem(x,y,2),DTem(x,y,3);...
                   DTem(x,y,2),DTem(x,y,4),DTem(x,y,5);...
                   DTem(x,y,3),DTem(x,y,5),DTem(x,y,6)];
            dti = Mask(x,y,z).*dti;
            [Vec,Val] = eig(dti);
            Lambda = scale_factor(x,y,z).*Val;
            tem = Vec*Lambda*Vec';
            % Diagonal element
            CndTensor(x,y,1,z) = abs(tem(1,1)); 
            CndTensor(x,y,4,z) = abs(tem(2,2)); 
            CndTensor(x,y,6,z) = abs(tem(3,3));
            % Off-diagonal element
            CndTensor(x,y,2,z) = tem(1,2);      
            CndTensor(x,y,3,z) = tem(1,3);      
            CndTensor(x,y,5,z) = tem(2,3);
            clear dti Vec Val tem Lambda;
        end
    end
    clear Dtem;
end
CndTensor(isnan(CndTensor)|isinf(CndTensor)) = 0;
end


