function [C,ScaleFactor] = mrci_dw_J_substitution(gradU,J,D,recon_parameters)
% This function is the part of magnetic resonance conductivity imaging (mrci)
% toolbox calulating the anisotropic conductivity using difussion-weighted 
% J-substitution algorithm. 
% For details of the algorithm please see the below reference,
% Jeong W. C. et al. IEEE Trans. Med. Imaging, 36, 2017.
% Writen by Saurav Z. K. Sajib, PhD. 
% Last date of modify [yyyy-mm-dd]: 2017-01-24.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% If you use this function please cite the below article:
% Sajib S. Z. K. et al, "Software Toolbox for Low-frequency Conductivity
% and Current Density Imaging using MRI," article submitted to IEEE Trans. Biomed. Eng.
% For any suggestion feel free to contact at iirc.mrci@gmail.com.
% Inputs of the function is,
% gradU[MNNs3E]: calculated gradient of voltage where,
% ux = squeeze(gradU(:,:,:,1,:));
% uy = squeeze(gradU(:,:,:,2,:)); 
% uz = squeeze(gradU(:,:,:,3,:));
% J[MNNs3E]: estimated current density image where, 
% Jx = squeeze(J(:,:,:,1,:)); 
% Jy = squeeze(J(:,:,:,2,:));
% Jz = squeeze(J(:,:,:,3,:));
% D[MN6Ns]: measured diffusion tensor image.
% recon_parameters.Mask[MN]: region of interest for (optional).
% Default value is 0.
% Output of the function is,
% C[MN6Ns]: reconstructed conductivity tensor image.
% ScaleFactor[MNNs]: reconstructed scale factor image.
%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
if isempty(recon_parameters.Mask)
    Mask = double(squeeze(gradU(:,:,:,1,1))~=0);
else 
    Mask = recon_parameters.Mask;
end
imSize  = size(Mask,1); 
Nslices = size(Mask,3); 
%%%%%%%%%%%%%%%%%% Reconstructing the conductivity tensor %%%%%%%%%%%%%%%%%
ux = squeeze(gradU(:,:,:,1,:));
uy = squeeze(gradU(:,:,:,2,:)); 
uz = squeeze(gradU(:,:,:,3,:));
Jx = squeeze(J(:,:,:,1,:)); 
Jy = squeeze(J(:,:,:,2,:));
Jz = squeeze(J(:,:,:,3,:));
clear gradU J;

ScaleFactor = zeros(imSize,imSize,Nslices);
for z = 1:Nslices
    
    JxTem = squeeze(Jx(:,:,z,:)); 
    JyTem = squeeze(Jy(:,:,z,:)); 
    JzTem = squeeze(Jz(:,:,z,:));
    
    uxTem = squeeze(ux(:,:,z,:)); 
    uyTem = squeeze(uy(:,:,z,:)); 
    uzTem = squeeze(uz(:,:,z,:));
    
    DTem = squeeze(D(:,:,:,z));
    
    for x = 1:imSize
        for y = 1:imSize
            if Mask(x,y,z) == 1
                dti = [DTem(x,y,1),DTem(x,y,2),DTem(x,y,3);...
                       DTem(x,y,2),DTem(x,y,4),DTem(x,y,5);...
                       DTem(x,y,3),DTem(x,y,5),DTem(x,y,6)];
                dti = Mask(x,y,z).*dti;
                gradU   = [squeeze(uxTem(x,y,:)),squeeze(uyTem(x,y,:)),squeeze(uzTem(x,y,:))]';
                J = [squeeze(JxTem(x,y,:)),squeeze(JyTem(x,y,:)),squeeze(JzTem(x,y,:))]';
                gradU_D = dti*gradU;
                num = J'*gradU_D;       num = diag(num); num = sum(num);
                den = gradU_D'*gradU_D; den = diag(den); den = sum(den);
                tem = num/den;
                tem(isnan(tem)|isinf(tem)) = 0;
                ScaleFactor(x,y,z) = -1*tem;
                clear tem dti gradU J gradU_D;
            end
        end
    end
    clear JxTem JyTem JzTem uxTem uyTem uzTem DTem;
end

C = conductivity_tensor(D,ScaleFactor,Mask);

end

function CndTensor = conductivity_tensor(D,scale_factor,Mask)
imSize = size(Mask,1);
Nslices = size(Mask,3);
size_scale_factor = size(scale_factor,1);
if size_scale_factor == 1
    scale_factor = scale_factor.*ones(imSize,imSize,Nslices);
end
scale_factor = Mask.*scale_factor;
%%%%%%%%%%%%%%%%%%%%%%%%%%% Conductivity tensor %%%%%%%%%%%%%%%%%%%%%%%%%%%
CndTensor = zeros(size(D));
for z = 1:Nslices
    DTem = squeeze(D(:,:,:,z));
    for x = 1:imSize
        for y = 1:imSize
            dti = [DTem(x,y,1),DTem(x,y,2),DTem(x,y,3);...
                   DTem(x,y,2),DTem(x,y,4),DTem(x,y,5);...
                   DTem(x,y,3),DTem(x,y,5),DTem(x,y,6)];
            dti = Mask(x,y,z).*dti;
            [Vec,Val] = eig(dti);
            Lambda = scale_factor(x,y,z).*Val;
            tem = Vec*Lambda*Vec';
            % Diagonal element
            CndTensor(x,y,1,z) = abs(tem(1,1)); 
            CndTensor(x,y,4,z) = abs(tem(2,2)); 
            CndTensor(x,y,6,z) = abs(tem(3,3));
            % Off-diagonal element
            CndTensor(x,y,2,z) = tem(1,2);      
            CndTensor(x,y,3,z) = tem(1,3);      
            CndTensor(x,y,5,z) = tem(2,3);
            clear dti Vec Val tem Lambda;
        end
    end
    clear Dtem;
end
CndTensor(isnan(CndTensor)|isinf(CndTensor)) = 0;
end


