function sigmaH = mrci_ept(B1, MR, recon_parameters)
% This function is the part of magnetic resonance conductivity imaging (mrci)
% toolbox calulating the projected current density (type-1) in MREIT. 
% For details of the algorithm please see the below reference,
% Lee J. et al. Magn. Res. Med., 76, 2016.  
% Writen by Saurav Z. K. Sajib, PhD.
% Modified and corrected by Prof. Oh I. Kwon, PhD.
% Last date of modify [yyyy-mm-dd]: 2017-01-20.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% If you use this function please cite the below article:
% Sajib S. Z. K. et al, "Software Toolbox for Low-frequency Conductivity
% and Current Density Imaging using MRI," article submitted to IEEE Trans. Biomed. Eng.
% For any suggestion feel free to contact at iirc.mrci@gmail.com.
% Inputs of the function is,
% B1[MNNs]: measured phase of the B1 map.
% MR[MNNs]: MR magnitude images to calculate weight.
% recon_parameters.VoxelSize[13]: voxel size in millimeter.
% recon_parameters.KernelSize[13]: Gaussian kernel size.
% recon_parameters.KernelVariance: Variance of Gaussian kernel.
% recon_parameters.Mask[MN]: region of interest for (optional).
% recon_parameters.B0: main field strength e.g. 3.0 T(optional).
% Deafault is 3.0T
% Output of the function is,
% sigmaH[MNNs]: reconstructed multi-slice high frequency conductivity images.

%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
if isempty(recon_parameters.Mask)
    Mask = double(MR>0.1*max(MR(:))&MR~=0);
    for i = 1:size(Mask,3)
        Mask(:,:,i) = imopen(Mask(:,:,i),strel('disk',2));
    end
    clear i;
else 
    Mask = recon_parameters.Mask;
end
VoxelSize = recon_parameters.VoxelSize;
%%%%%%%%%%%%%%%%%%%%%%%%%%% Checking the inputs %%%%%%%%%%%%%%%%%%%%%%%%%%%
SizeB1 = size(B1);
SizeMR = size(MR);
if SizeB1 ~= SizeMR
    error('Input data size must be same [MNNs]');
end

%%%%%%%%%%%%%%%%%%%%% Reconstructing the conductivity %%%%%%%%%%%%%%%%%%%%%
KernelSize = recon_parameters.KernelSize;
KernelVariance = recon_parameters.KernelVariance;
mu0     = 4*pi*1e-7;    % Free space permeability   
gamma   = 42.576e6;     % Gyromagnetic ratio
omega   = 2*pi*gamma*recon_parameters.B0; % Lamor frequency @ 3T
scale   = 2*mu0*omega;
if size(B1,3) == 1
    disp('Calcualting 2D Laplacian.');
    lap_phi = mrept_laplacian_2D(B1,MR,VoxelSize(1:2),KernelSize(1:2),KernelVariance,Mask);
elseif size(B1,3) >1
    disp('Calcualting 3D Laplacian.');
    lap_phi = mrept_laplacian_3D(B1,MR,VoxelSize,KernelSize,KernelVariance,Mask);
end
    sigmaH = lap_phi/scale;
end

function lap_phi = mrept_laplacian_3D(phase,mr,VoxelSize,KernelSize,KernelVar,roi)
% Zero padding
N = size(phase);
PaddZerosSize = 2*KernelSize+1;
TemMR = zeros(N(1)+PaddZerosSize(1),...
              N(2)+PaddZerosSize(2),...
              N(3)+PaddZerosSize(3));
TemMR(KernelSize(1)+1:N(1)+KernelSize(1),...
      KernelSize(2)+1:N(2)+KernelSize(2),...
      KernelSize(3)+1:N(3)+KernelSize(3)) = mr; 
TemPhase = zeros(N(1)+PaddZerosSize(1),...
                 N(2)+PaddZerosSize(2),...
                 N(3)+PaddZerosSize(3));
TemPhase(KernelSize(1)+1:N(1)+KernelSize(1),...
         KernelSize(2)+1:N(2)+KernelSize(2),...
         KernelSize(3)+1:N(3)+KernelSize(3)) = phase;
Temroi = zeros(N(1)+PaddZerosSize(1),...
              N(2)+PaddZerosSize(2),...
              N(3)+PaddZerosSize(3));
Temroi(KernelSize(1)+1:N(1)+KernelSize(1),...
      KernelSize(2)+1:N(2)+KernelSize(2),...
      KernelSize(3)+1:N(3)+KernelSize(3)) = roi; 
     
clear mr phase roi;
mr      = TemMR;        
phase   = TemPhase; 
roi = Temroi;
clear TemMR Temphase Temroi;
% Mesh griding
x = (-N(1)-KernelSize(1):N(1)+KernelSize(1))*VoxelSize(1);
y = (-N(2)-KernelSize(2):N(2)+KernelSize(2))*VoxelSize(2); y = rot90(y);
z = (-N(3)-KernelSize(3):N(3)+KernelSize(3))*VoxelSize(3);
[x,y,z] = meshgrid(x,y,z);
% Calulating 3D laplacian of phase using 2nd order weighted polynomial phase fitting 
S = roi.*mr;
phi = roi.*phase;
clear mr phase;
lap_phi = zeros(N(1)+PaddZerosSize(1),...
                N(2)+PaddZerosSize(2),...
                N(3)+PaddZerosSize(3));
h = waitbar(0,'Second-order polynomial fitting...');
set(h,'Name','Second-order polynomial fitting progress');
for i = 1+KernelSize(1):N(1)+KernelSize(1)
    for j = 1+KernelSize(2):N(2)+KernelSize(2)
        for k = 1+KernelSize(3):N(3)+KernelSize(3)
            if roi(i,j,k)==1
                STem = S(i-KernelSize(1):i+KernelSize(1),...
                         j-KernelSize(2):j+KernelSize(2),...
                         k-KernelSize(3):k+KernelSize(3));
                phaseTem = phi(i-KernelSize(1):i+KernelSize(1),...
                               j-KernelSize(2):j+KernelSize(2),...
                               k-KernelSize(3):k+KernelSize(3));
                phaseTem = phaseTem(:);
                u = x(i-KernelSize(1):i+KernelSize(1),...
                      j-KernelSize(2):j+KernelSize(2),...
                      k-KernelSize(3):k+KernelSize(3));
                v = y(i-KernelSize(1):i+KernelSize(1),...
                      j-KernelSize(2):j+KernelSize(2),...
                      k-KernelSize(3):k+KernelSize(3));
                w = z(i-KernelSize(1):i+KernelSize(1),...
                      j-KernelSize(2):j+KernelSize(2),...
                      k-KernelSize(3):k+KernelSize(3));  
                u = u(:); v = v(:); w = w(:);
                A = [ones(size(u,1),1),u,v,w,u.*v,v.*w,w.*u,u.^2,v.^2,w.^2];
                wg  = exp(-((STem-STem(KernelSize(1)+1,KernelSize(2)+1,KernelSize(3)+1))).^2./(2* KernelVar)^2);
                wg  = diag(wg(:));
                c = pinv(A'*wg*A)*(A'*wg*phaseTem);
                lap_phiTem = 2*(c(end-2)+c(end-1)+c(end)); 
                lap_phi(i,j,k) = lap_phiTem ;
            end
        end
        h1 = fix(100*(i/(N(1)+KernelSize(1))));
        waitbar(i/(N(1)+KernelSize(1)),h,strcat(num2str(h1),'%'));
        clear h1;
    end
end
lap_phi(isnan(lap_phi)|isinf(lap_phi)) =  0;
lap_phi1 = lap_phi(1+KernelSize(1):N(1)+KernelSize(1),...
                   1+KernelSize(2):N(2)+KernelSize(2),...
                   1+KernelSize(3):N(3)+KernelSize(3));
clear lap_phi;
lap_phi = lap_phi1;
clear lap_phi1;
close(h);
end

function lap_phi = mrept_laplacian_2D(phase,mr,VoxelSize,KernelSize,KernelVar,roi)
% Imaging parameters 
N = size(phase);
% Zero padding
PaddZerosSize = 2*KernelSize+1;
TemMR = zeros(N(1)+PaddZerosSize(1),...
              N(2)+PaddZerosSize(2)); 
TemMR(KernelSize(1)+1:N(1)+KernelSize(1),...
      KernelSize(2)+1:N(2)+KernelSize(2)) = mr; 
TemPhase = zeros(N(1)+PaddZerosSize(1),...
                 N(2)+PaddZerosSize(2));
TemPhase(KernelSize(1)+1:N(1)+KernelSize(1),...
         KernelSize(2)+1:N(2)+KernelSize(2))= phase; 
Temroi = zeros(N(1)+PaddZerosSize(1),...
              N(2)+PaddZerosSize(2)); 
Temroi(KernelSize(1)+1:N(1)+KernelSize(1),...
      KernelSize(2)+1:N(2)+KernelSize(2)) = roi; 
     
clear mr phase roi;
mr = TemMR; 
phase = TemPhase; 
roi = Temroi;
clear TemMR TemPhase Temroi;
% Mesh griding
x = (-N(1)-KernelSize(1):N(1)+KernelSize(1))*VoxelSize(1);
y = (-N(2)-KernelSize(2):N(2)+KernelSize(2))*VoxelSize(2);
y = rot90(y);
[x,y] = meshgrid(x,y);
clear PixelSizeX PixelSizeY;

% Calulating 2D laplacian of phase using 2nd order weighted polynomial phase fitting 
S = roi.*mr;
phi = roi.*phase;
clear mr phase;
lap_phi = zeros(N(1)+PaddZerosSize(1),N(2)+PaddZerosSize(2));
h = waitbar(0,'Second-order polynomial fitting...');
set(h,'Name','Second-order polynomial fitting progress');

for i = 1+KernelSize(1):N(1)+KernelSize(1)
    for j = 1+KernelSize(2):N(2)+KernelSize(2)
        if roi(i,j) == 1
            STem = S(i-KernelSize(1):i+KernelSize(1),...
                     j-KernelSize(2):j+KernelSize(2));
            phaseTem = phi(i-KernelSize(1):i+KernelSize(1),...
                           j-KernelSize(2):j+KernelSize(2));
            phaseTem = phaseTem(:);
            u = x(i-KernelSize(1):i+KernelSize(1),....
                  j-KernelSize(2):j+KernelSize(2));
            v = y(i-KernelSize(1):i+KernelSize(1),...
                  j-KernelSize(2):j+KernelSize(2));
            u = u(:); v = v(:);
            A = [ones(size(u,1),1),u,v,u.*v,u.^2,v.^2];
            wg  = exp(-(abs(STem-STem(KernelSize(1)+1,KernelSize(2)+1))).^2/(2* KernelVar)^2);
            wg  = diag(wg(:));
            c = pinv(A'*wg*A)*A'*wg*phaseTem;
            lap_phiTem = 2*(c(end-1)+c(end));
            lap_phi(i,j) = lap_phiTem ;
        end
    end
    h1 = fix(100*(i/(N(1)+KernelSize(1))));
    waitbar(i/(N(1)+KernelSize(1)),h,strcat(num2str(h1),'%'));
    clear h1;
end
lap_phi(isnan(lap_phi)|isinf(lap_phi)) =  0;
lap_phi1 = lap_phi(1+KernelSize(1):N(1)+KernelSize(1),1+KernelSize(2):N(2)+KernelSize(2));
clear lap_phi;
lap_phi = lap_phi1;
clear lap_phi1;
close(h);
end