function recon_parameters = reconstruction_parameters(Algorithm,varargin)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This function is the part of magnetic resonance conductivity imaging (mrci) toolbox
% Last date of modify [yyyy-mm-dd]: 2017-01-23.
% Copyright (C) to Impedance Imaging Research Center (IIRC).
% For any suggestion feel free to contact at iirc.mrci@gmail.com.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Algorithms are:
% Current density
%  01. Projected current density type-1
%  02. Projected current density type-2
%  03. Regional projected current density
% Isotropic/ equivelent isotropic conductivity 
%  04. Harmonic Bz
%  05. J subtitution
%  06. Transversal J substitution
% Anisotropic conductivity using DT-MREIT
%  07. DTMREIT noniterative 
%  08. Dw J subsutitution 
% Anisotropic conductivity using CTI
%  09. CTI
% MR-EPT
%  10. EPT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Variable input list are,
%  01.VoxelSize[13], in millimeter (mandatory input most of the algorithm)
%  02.Mask[MN], region of interest
%  03.s0, boundary condition for solvong Poisson equation
%  04.Lambda, Tikonhov regularization parameters (used for harmonic Bz, noniterative harmonic Bz, DT-MREIT nonitertive algorithms) 
%  06.PixelSmoothFactor, value for boundary smoothing (used to solve the Poisson equation) 
%  07.sigma0[MN], reconstructed/ or homogeneous value of conductivity used in transversal J substitution algorithm
%  08.scalefactor0, boundary condition for solvong Poisson equation (used in DT-MREIT nonitertive algorithm) 
%  09.KernelSize[13], size of kernel used in EPT function.
%  10.KernelVariance: variance of Gaussian kernel used in EPT function.
%  11.B0: main magnent field strength in T
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

recon_parameters = struct;
recon_parameters.Algorithm = lower(Algorithm);
recon_parameters.VoxelSize = [];
recon_parameters.Mask = [];
recon_parameters.s0 = [];
recon_parameters.Lambda = [];
recon_parameters.PixelSmoothFactor = [];
recon_parameters.sigma0 = [];
recon_parameters.scalefactor0 = [];
recon_parameters.KernelSize = [];
recon_parameters.KernelVariance = [];
recon_parameters.B0 = [];

Narg = nargin;
if rem(Narg-1,2) ~=0
    error('Check the numbers of input arguments.');
end

switch recon_parameters.Algorithm
    
    % Projected current density type-1
    case 'projected current density type-1'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'VoxelSize')
                recon_parameters.VoxelSize = varargin{i}*1e-3;
            elseif strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            end
        end
        if isempty(recon_parameters.VoxelSize)
            error('This algorithm needs the voxel size in millimeter as a mandatory input.');
        end

    % Projected current density type-2
    case 'projected current density type-2'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'VoxelSize')
                recon_parameters.VoxelSize = varargin{i}*1e-3;
            elseif strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            end
        end
        if isempty(recon_parameters.VoxelSize)
            error('This algorithm needs the voxel size in millimeter as a mandatory input.');
        end

    % Regional projected current density
    case 'regional projected current density'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'VoxelSize')
                recon_parameters.VoxelSize = varargin{i};
            elseif strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            end
        end
        if isempty(recon_parameters.VoxelSize)
            error('This algorithm needs the voxel size in millimeter as a mandatory input.');
        end

    % Harmonic Bz algorithm   
    case 'harmonic bz'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'VoxelSize')
                recon_parameters.VoxelSize = varargin{i}*1e-3;
            elseif strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            elseif strcmpi(VariableList,'s0')
                recon_parameters.s0 = varargin{i};
            elseif strcmpi(VariableList,'Lambda')
                recon_parameters.Lambda = varargin{i};  
            elseif strcmpi(VariableList,'PixelSmoothFactor')
                recon_parameters.PixelSmoothFactor = varargin{i};
            end
        end
        if isempty(recon_parameters.VoxelSize)
            error('This algorithm needs the voxel size in millimeter as a mandatory input.');
        elseif isempty(recon_parameters.s0)
            recon_parameters.s0 = 1;
        elseif isempty(recon_parameters.Lambda)
            recon_parameters.Lambda = 1;
        elseif isempty(recon_parameters.PixelSmoothFactor)
            recon_parameters.PixelSmoothFactor = 0;          
        end
        
    % J substitution
    case 'j substitution'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            end
        end   
        
    % Transversal J substitution
    case 'transversal j substitution'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'VoxelSize')
                recon_parameters.VoxelSize = varargin{i}*1e-3;
            elseif strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            elseif strcmpi(VariableList,'sigma0')
                recon_parameters.sigma0 = varargin{i};
            end
        end
        if isempty(recon_parameters.VoxelSize)
            error('This algorithm needs the voxel size in millimeter as a mandatory input.');
        elseif isempty(recon_parameters.sigma0)
            recon_parameters.sigma0 = 1;
        end
        
    % Noniterative harmonic Bz
    case 'noniterative harmonic bz'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'VoxelSize')
                recon_parameters.VoxelSize = varargin{i}*1e-3;
            elseif strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            elseif strcmpi(VariableList,'s0')
                recon_parameters.s0 = varargin{i};
            elseif strcmpi(VariableList,'Lambda')
                recon_parameters.Lambda = varargin{i};  
            elseif strcmpi(VariableList,'PixelSmoothFactor')
                recon_parameters.PixelSmoothFactor = varargin{i};                
            end
        end
        if isempty(recon_parameters.VoxelSize)
            error('This algorithm needs the voxel size in millimeter as a mandatory input.');
        elseif isempty(recon_parameters.s0)
            recon_parameters.s0 = 1;
        elseif isempty(recon_parameters.Lambda)
            recon_parameters.Lambda = 1;
        elseif isempty(recon_parameters.PixelSmoothFactor)
            recon_parameters.PixelSmoothFactor = 0;                      
        end
        
    % DT-MREIT noniterative
    case 'dtmreit noniterative'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'VoxelSize')
                recon_parameters.VoxelSize = varargin{i}*1e-3;
            elseif strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            elseif strcmpi(VariableList,'scalefactor0')
                recon_parameters.scalefactor0 = varargin{i};
            elseif strcmpi(VariableList,'Lambda')
                recon_parameters.Lambda = varargin{i};
            elseif strcmpi(VariableList,'PixelSmoothFactor')
                recon_parameters.PixelSmoothFactor = varargin{i};
            end
        end
        if isempty(recon_parameters.VoxelSize)
            error('This algorithm needs the voxel size in millimeter as a mandatory input.');
        elseif isempty(recon_parameters.scalefactor0)
            recon_parameters.scalefactor0 = 1;
        elseif isempty(recon_parameters.Lambda)
            recon_parameters.Lambda = 1;
        elseif isempty(recon_parameters.PixelSmoothFactor)
            recon_parameters.PixelSmoothFactor = 0;
        end
    % DW J substitution
    case 'dw j substitution'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            end
        end
        
    % CTI
    case 'cti'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            end
        end
        
    % EPT
    case 'ept'
        for i = 2:2:Narg
            VariableList = varargin{i-1};
            if strcmpi(VariableList,'VoxelSize')
                recon_parameters.VoxelSize = varargin{i}*1e-3;
            elseif strcmpi(VariableList,'Mask')
                recon_parameters.Mask = varargin{i};
            elseif strcmpi(VariableList,'KernelSize')
                recon_parameters.KernelSize = varargin{i};
            elseif strcmpi(VariableList,'KernelVariance')
                recon_parameters.KernelVariance = varargin{i};  
           elseif strcmpi(VariableList,'B0')
                recon_parameters.B0 = varargin{i};       
           end
        end
        if isempty(recon_parameters.VoxelSize)
            error('This algorithm needs the voxel size in millimeter as a mandatory input.');
        elseif isempty(recon_parameters.KernelSize)
            error('This algorithm needs the kernel size as a mandatory input.');
        elseif isempty(recon_parameters.KernelVariance)
            error('This algorithm needs the Gaussian kernel variance as a mandatory input.');
         elseif isempty(recon_parameters.B0)
            recon_parameters.B0 = 3;    
        end
        
    otherwise
            disp('Algorithm is not specifiied. To see the list of algorithms please type help reconstruction_parameters.');
end        
    


